<?php
/**
 * Plugin Name: Enable Safe SVG (Lite)
 * Description: Permite subir SVGs con saneamiento básico y control de permisos.
 * Version: 1.0.0
 * Author: Jonathan Alvizures
 * License: GPL-2.0+
 */

if ( ! defined('ABSPATH') ) exit;

class Enable_Safe_SVG_Lite {
    const CAP = 'upload_svg';

    public function __construct() {
        // Permisos/capability para quién puede subir SVG
        register_activation_hook(__FILE__, [$this, 'on_activate']);
        register_deactivation_hook(__FILE__, [$this, 'on_deactivate']);

        // Permitir MIME SVG
        add_filter('upload_mimes', [$this, 'allow_svg_mime']);

        // Validación de extensión/MIME en WordPress 5.1+
        add_filter('wp_check_filetype_and_ext', [$this, 'check_filetype_and_ext'], 10, 4);

        // Saneamiento antes de guardar
        add_filter('wp_handle_upload_prefilter', [$this, 'sanitize_svg_prefilter']);

        // Mostrar bien las miniaturas SVG en la librería
        add_action('admin_head', [$this, 'admin_css_fix']);
    }

    public function on_activate() {
        $roles = ['administrator','editor'];
        foreach ($roles as $role_name) {
            $role = get_role($role_name);
            if ($role && ! $role->has_cap(self::CAP)) {
                $role->add_cap(self::CAP);
            }
        }
    }

    public function on_deactivate() {
        // Opcional: no retiramos la capability para no romper flujos al reactivar.
    }

    public function allow_svg_mime($mimes) {
        // Solo permitir si el usuario tiene la capability definida
        if ( current_user_can(self::CAP) ) {
            $mimes['svg']  = 'image/svg+xml';
            $mimes['svgz'] = 'image/svg+xml';
        }
        return $mimes;
    }

    public function check_filetype_and_ext($data, $file, $filename, $mimes) {
        $ext = pathinfo($filename, PATHINFO_EXTENSION);

        if ( in_array(strtolower($ext), ['svg', 'svgz'], true) ) {
            // Fuerza MIME correcto si el usuario tiene permisos
            if ( current_user_can(self::CAP) ) {
                $data['ext']  = 'svg';
                $data['type'] = 'image/svg+xml';
            } else {
                // Bloquear si no tiene permisos
                $data['ext']  = false;
                $data['type'] = false;
            }
        }
        return $data;
    }

    public function sanitize_svg_prefilter($file) {
        $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        if ( ! in_array($ext, ['svg','svgz'], true) ) return $file;

        // Permisos
        if ( ! current_user_can(self::CAP) ) {
            $file['error'] = __('No tienes permisos para subir SVG.', 'enable-safe-svg');
            return $file;
        }

        // Leer contenido
        $svg = @file_get_contents($file['tmp_name']);
        if ($svg === false || trim($svg) === '') {
            $file['error'] = __('El archivo SVG está vacío o no se pudo leer.', 'enable-safe-svg');
            return $file;
        }

        // Tamaño razonable (evitar payloads enormes). Opcional: 2 MB
        $max_bytes = 2 * 1024 * 1024;
        if ( filesize($file['tmp_name']) > $max_bytes ) {
            $file['error'] = sprintf(__('El SVG es demasiado grande (máx %d MB).', 'enable-safe-svg'), 2);
            return $file;
        }

        // Saneamiento básico con DOMDocument: eliminar <script>, <foreignObject> y atributos on*
        // Nota: Esto es un filtro "lite". Para entornos críticos, usa una librería dedicada como en el plugin "Safe SVG".
        $clean = $this->basic_sanitize_svg($svg);
        if ($clean === null) {
            $file['error'] = __('El SVG no es válido o contiene nodos inseguros y no pudo sanearse.', 'enable-safe-svg');
            return $file;
        }

        // Sobrescribir el tmp con la versión saneada
        @file_put_contents($file['tmp_name'], $clean);
        return $file;
    }

    private function basic_sanitize_svg($svg) {
        // Rápido rechazo: debe contener <svg
        if ( stripos($svg, '<svg') === false ) return null;

        // Evitar entidades externas
        libxml_use_internal_errors(true);

        $dom = new DOMDocument();
        // Cargar restringiendo entidades externas
        $loaded = $dom->loadXML($svg, LIBXML_NONET | LIBXML_NOENT | LIBXML_COMPACT | LIBXML_NOBLANKS);
        if (!$loaded) return null;

        $xpath = new DOMXPath($dom);

        // Eliminar <script> y <foreignObject>
        foreach (['script','foreignObject'] as $tag) {
            foreach ($xpath->query('//' . $tag) as $node) {
                $node->parentNode->removeChild($node);
            }
        }

        // Eliminar atributos on* (onload, onclick, etc.)
        $nodes = $xpath->query('//*');
        foreach ($nodes as $node) {
            if (!$node->attributes) continue;
            $toRemove = [];
            foreach ($node->attributes as $attr) {
                $name = strtolower($attr->name);
                // on* o xlink:href con javascript:
                if (strpos($name, 'on') === 0) {
                    $toRemove[] = $attr->name;
                    continue;
                }
                if (in_array($name, ['href','xlink:href'], true)) {
                    if (preg_match('/^\s*javascript:/i', $attr->value)) {
                        $toRemove[] = $attr->name;
                        continue;
                    }
                }
                // style con url("javascript:...") o -moz-binding
                if ($name === 'style') {
                    $val = strtolower($attr->value);
                    if (strpos($val, 'javascript:') !== false || strpos($val, '-moz-binding') !== false) {
                        $toRemove[] = $attr->name;
                    }
                }
            }
            foreach ($toRemove as $attrName) {
                $node->removeAttribute($attrName);
            }
        }

        // Asegurar MIME: añadir xmlns si falta
        $svgEls = $dom->getElementsByTagName('svg');
        if ($svgEls->length > 0) {
            $svgEl = $svgEls->item(0);
            if (!$svgEl->hasAttribute('xmlns')) {
                $svgEl->setAttribute('xmlns', 'http://www.w3.org/2000/svg');
            }
        }

        $out = $dom->saveXML($dom->documentElement);
        // Asegurar que la salida sea solo el nodo <svg> (sin prolog)
        if (! $out || stripos($out, '<svg') === false) return null;

        return $out;
    }

    public function admin_css_fix() {
        echo '<style>
            .attachment .thumbnail img[src$=".svg"],
            img[src$=".svg"].attachment-post-thumbnail,
            .media-modal .attachments-browser .attachment-preview img[src$=".svg"] {
                width: 100% !important;
                height: auto !important;
                max-height: 100% !important;
            }
        </style>';
    }
}

new Enable_Safe_SVG_Lite();